; Software License Agreement
;
; The software supplied herewith by Microchip Technology Incorporated (the "Company")
; for its PICmicro Microcontroller is intended and supplied to you, the Company's
; customer, for use solely and exclusively on Microchip PICmicro Microcontroller
; products.
;
; The software is owned by the Company and/or its supplier, and is protected under
; applicable copyright laws. All rights are reserved. Any use in violation of the
; foregoing restrictions may subject the user to criminal sanctions under applicable
; laws, as well as to civil liability for the breach of the terms and conditions of
; this license.
;
; THIS SOFTWARE IS PROVIDED IN AN "AS IS" CONDITION. NO WARRANTIES, WHETHER EXPRESS,
; IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED TO, IMPLIED WARRANTIES OF
; MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE.
; THE COMPANY SHALL NOT, IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL
; OR CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.

;****************************************************************************
;*  Filename: APPNOTE.ASM
;****************************************************************************
;*	Author: 		Brett Duane
;*	Company:		Microchip Technology
;*	Revision:		Rev 1.0
;*	Date:		3-9-99
;*	Assembled using MPASM rev 4.00.00
;****************************************************************************
;* 	Include files:	p16c17a.inc	Rev 1.01
;****************************************************************************
;*		
;*	Switching buck regulator using PIC16C17a using A/D, 
;*	PWM (CCP) and timer2 modules.
;*
;*	PID control loop implementation. 
;*	Executes 4883 loops per second.
;*	Spends most of its time looping waiting for timer2 overflows.
;*
;*	A/D inputs and PWM output are 8 bits, 
;*	with internal calculations done in 16 bits
;*
;*	Timer2 postscaler (1:16) generates an interrupt.
;*
;*	RA0 converts a 0-5V input from trimmer, but is not used.
;*	RA1 monitors VOUT.
;*	RA2 is the up voltage push button input.
;*	RA3 monitors VUNREG.
;*	RA5 is the down voltage push button input.
;*
;*	RB<5:0> drives LEDs to indicate the output voltage.
;*	RB<7:6> drives LEDs to indicate overload or shutdown.
;*
;*	RC2 is the PWM source for the switching converter.
;*
;*
;*	Configuration Bit Settings
;*		Brown-out detect is off
;*		Code protect is off
;*		Power-up timer is enabled
;*		Watchdog timer is disabled
;*		10MHz resonator is driven in XT mode
;*		
;*	Program Memory Words Used:   230
;*	Program Memory Words Free:  1818
;*	
;*	Data Memory Bytes Used:       24
;*	Data Memory Bytes Free:      104
;*	
;****************************************************************************
;*  What's Changed
;*
;*	Date	Description of Change
;*
;*	3-3-99	This is the initial release.
;*
;****************************************************************************

	list p=16c72a
	#include	<p16c72a.inc>

	__config _BODEN_OFF & _CP_OFF & _PWRTE_ON & _WDT_OFF & _XT_OSC


	cblock	0x020

		UPCL	;up button debounce
		UPCH	;up button debounce
		DNCL	;down button debounce
		DNCH	;down button debounce
		SETPOINT	;voltage setpoint - RA0 result (unsigned)
		VOUT	;output voltage feedback - RA1 result (unsigned)
		VUNREG	;source voltage feedback - RA3 result (unsigned)
		TEMPA	;temp variable
		TEMPB	;temp variable
		INT	;integral componant
		INTH	;integral componant
		PRO	;proportional componant
		PROH	;proportional componant
		DIF	;difference componant
		DIFH	;difference componant
		PWM	;PWM drive
		PWMH	;PWM drive
		E0	;present error
		E0H	;present error
		E1	;past error
		E1H	;past error
		T2POST	;postscaler interrupt counter
		ISRS	;isr variable
		ISRW	;isr variable

	endc

DEL1	equ	0xf9	;text equate - debounce delay (Fosc=10MHz)
AVOUT	equ	0x89	;text equate - select feedback channel
AVUNREG	equ	0x99	;text equate - select Vunrg channel

;****************************************************************************
;*	Reset vector
;*		This is the reset vector
;*		
;****************************************************************************

	org	0x00	;reset vector
	goto	Main	;program start


;****************************************************************************
;*	INTERRUPT SERVICE ROUTINE
;*		This ISR counts timer2 interrupts
;*		
;*	Input Variables:
;*		T2POST	Counts overflow interrupts
;*
;*	Output Variables:
;*		T2POST	Counts overflow interrupts
;*
;****************************************************************************

	org	0x04	;INTerrupt service routine

	movwf	ISRW	;save W
	swapf	STATUS,W	;get status
	movwf	ISRS	;save status

	bcf	PIR1,TMR2IF	;clear INTerrupt request flag

	incf	T2POST,F	;increment INTerrupt counter

	swapf	ISRS,W	;get status
	movwf	STATUS	;restore status
	swapf	ISRW,F	;restore W
	swapf	ISRW,W	;restore W

	retfie		;return from INTerrupt	

;****************************************************************************
;*	Main	Begining of main loop
;*		First segment of code initializes controller
;****************************************************************************

Main
	bcf	STATUS,RP1	;bank1 initialization
	bsf	STATUS,RP0	;bank1
;adc
	movlw	0x04	;RA0,1,3 analog, RA2,5 digital, Vref=Vdd
	movwf	ADCON1

	movlw	0x2f	;RA0,1,3 analog in; RA2,5 digital in, RA4 dig out
	movwf	TRISA
;PWM
	movlw	0xFF	;set portC to all inputs
	movwf	TRISC
	bcf	TRISC,2	;make RC2/PWM output

	movlw	.63	;PWM period =  78.125KHz (full  8 bit resolution)
	movwf	PR2

;timer2
	bsf	PIE1,TMR2IE	;enable timer2 postscaler INTerrupts

;display
	clrf	TRISB	;make PORTB all outputs

	bcf	STATUS,RP0	;select bank0
;adc
	movlw	AVOUT	;(10MHz osc) set A/D conv clock (Fosc/32), 
	movwf	ADCON0	;select RA1(AN1), turn on A/D
;PWM
	bcf	CCP1CON,4	;clear ls bits of PWM duty cycle
	bcf	CCP1CON,5
	clrf	CCPR1L	;set PWM to 0 (turn off PWM)

	movlw	0x04	;enable timer2, set prescale=1:1, postscale=1:1
	movwf	T2CON
	movlw	0x0C	;set CCP1 to PWM mode
	movwf	CCP1CON

;****************************************************************************
;*	Restart	Clears memory
;*
;*		Initializes display LEDs, desired output voltage, 
;*		debounce counters
;*
;*		Enables interrupts
;*		
;*	Output Variables:
;*		SETPOINT	desired output voltage, set to 3.0V
;*		DNCL, DNCH	Down voltage button debounce counter
;*		UPCL, UPCH	Up voltage button debounce counter
;*
;****************************************************************************

Restart	movlw	0x20	;clear memory from 0x20 to 0x3f
	movwf	FSR
ClrMem	clrf	INDF		
	incf	FSR,F
	btfss	FSR,6
	goto	ClrMem

	movlw	0x01	;light 3.0V LED
	movwf	PORTB

	movlw	0x4d	;initial 3.0V setting
	movwf	SETPOINT

	clrf	DNCL	;clear down button debounce counter
	clrf	UPCL	;clear up button debounce counter

	movlw	DEL1
	movwf	DNCH	;preset down button debounce counter high byte
	movwf	UPCH	;preset up button debounce counter high byte

	bsf	INTCON,PEIE	;enable peripheral INTerrupt sources
	bsf	INTCON,GIE	;enable all INTerrupts

;****************************************************************************
;*
;*	Again	Top of main loop
;*
;*		Waits for 8 timer2 interrupts
;*
;*		A/D converts VOUT
;*		
;*		Aquires VUNREG channel
;*		
;*	Input Variables:
;*		T2POST	Timer2 interrupt counter
;*
;*	Output Variables:
;*		VOUT	Conversion result
;*		
;****************************************************************************

Again
	btfss	T2POST,3	;long delay
	goto	Again	;try again

	clrf	T2POST	;clear counter

;--- start conversion - feedback
	bsf	ADCON0,GO_DONE	;start conversion
	nop
Wc2	btfsc	ADCON0,GO_DONE	;test if done
	goto	Wc2	;no, wait some more
	movf	ADRES,W	;get conversion result
	movwf	VOUT	;save result
;--- end conversion
	movlw	AVUNREG	;select VUNREG channel
	movwf	ADCON0

;****************************************************************************
;*	FErr	Finds difference (error) between SETPOINT and VOUT
;*		E0H:E0 = SETPOINT - VOUT
;*		
;*	Input Variables:
;*		SETPOINT	Desired output voltage
;*
;*	Output Variables:
;*		E0H:E0	Signed error
;****************************************************************************

FErr
	clrf	E0H	;clear error high byte

	movf	VOUT,W
	subwf	SETPOINT,W	;f-w=d
	movwf	E0	;save new error

	btfss	STATUS,C	;was there a borrow?
	comf	E0H,F	;yes

;****************************************************************************
;*	Ppp	Produces proportional componant by multiplying E0H:E0 by Kp
;*
;*		PROH:PRO = E0H:E0 * Kp
;*		Kp = 2^3 - Produced by 3 left shifts
;*
;*		This term forces the output close to the desired output quickly,
;*		but will never completely eliminate the error.
;*
;*	Input Variables:
;*		E0H:E0	Error found at top of loop
;*		Kp	Proportional gain factor (constant)
;*
;*	Output Variables:
;*		PROH:PRO	Proportional componant
;*
;****************************************************************************

Ppp
	movf	E0,W	;move E0 to temp space
	movwf	TEMPA
	movf	E0H,W
	movwf	TEMPB

	bcf	STATUS,C	;mult E0 by 2
	rlf	TEMPA,F
	rlf	TEMPB,F

	bcf	STATUS,C	;mult E0 by 2
	rlf	TEMPA,F
	rlf	TEMPB,F

	bcf	STATUS,C	;mult E0 by 2
	rlf	TEMPA,F
	rlf	TEMPB,F

PppD	movf	TEMPA,W	;move result in temp space to pro
	movwf	PRO
	movf	TEMPB,W
	movwf	PROH

;****************************************************************************
;*	DifCom	Computes differential componant
;*		
;*		Finds difference between this loop error and previous loop error
;*		DIFH:DIF = (E1H:E1 - E0H:E0) * Kd
;*		
;*		Kd = 2^3 - Produced by 3 left shifts
;*		
;*		This term tends to slow controller response.
;*		
;*	Input Variables:
;*		E1H:E1	Previous loop error
;*		E0H:E0	Present loop error
;*		Kd	Differential gain factor (constant)
;*
;*	Output Variables:
;*		DIFH:DIF	differential componant
;*
;****************************************************************************

DifCom
	movf	E1H,W		;get prev error high byte
	subwf	E0H,W		;f-w=d   E0-E1=w
	movwf	DIFH		;save difference high byte

	movf	E1,W		;get prev error low byte
	subwf	E0,W		;f-w=d   E0-E1=w
	movwf	DIF		;save difference low byte

	btfss	STATUS,C	;was there a borrow?
	decf	DIFH,F		;yes

	;allow difference to be modified here

	bcf	STATUS,C	;mult dif by 2
	rlf	DIF,F	
	rlf	DIFH,F

	bcf	STATUS,C	;mult dif by 2
	rlf	DIF,F	
	rlf	DIFH,F

	bcf	STATUS,C	;mult dif by 2
	rlf	DIF,F	
	rlf	DIFH,F



;****************************************************************************
;*	IntCom	Computes integral componant
;*
;*		Multiplies present error by Ki,
;*		adds result to INTH:INT
;*		
;*		INTH:INT = INTH:INT + E0H:E0 * Ki
;*		
;*		Ki = 2^3 - Produced by 3 left shifts
;*		
;*		This term will eliminate all error,
;*		but not quickly
;*		
;*	Input Variables:
;*		E0H:E0	Present loop error
;*		INTH:INT	Running total of errors
;*		Ki	Integral gain factor (constant)
;*
;*	Output Variables:
;*		DIFH:DIF	differential componant
;****************************************************************************

IntCom
	movf	E0H,W		;move E0 to temp space
	movwf	TEMPB
	movf	E0,W
	movwf	TEMPA

	;allow error to be modified (in temp space) here before adding to INTegral

	bcf	STATUS,C
	rlf	TEMPA,F		;eo
	rlf	TEMPB,F		;eoh

	bcf	STATUS,C
	rlf	TEMPA,F		;eo
	rlf	TEMPB,F		;eoh

	bcf	STATUS,C
	rlf	TEMPA,F		;eo
	rlf	TEMPB,F		;eoh

IntD	movf	TEMPA,W		;get current error, E0
	addwf	INT,F		;add to INT, store in INT
	btfsc	STATUS,C	;was there a carry?
	incf	INTH,F		;yes, inc INT high byte

	movf	TEMPB,W		;get E0 high byte, E0H
	addwf	INTH,F

;****************************************************************************
;*	Total	Sums proportional, integral, and differential terms
;*		
;*		PWMH:PWM = INTH:INT + PROH:PRO + DIFH:DIF
;*		
;*		If the result should ever go negative,
;*		the result is forced to 0, and the overload LED is turned on.
;*		(This is an error condition. We can't have a negative PWM.)
;*		
;*	Input Variables:
;*		INTH:INT	Integral term
;*		PROH:PRO	Proportional term
;*		DIFH:DIF	Differential term
;*		
;*	Output Variables:
;*		PWMH:PWM	Sum of terms
;****************************************************************************

Total	
PCom	movf	PROH,W	;add in proportional term
	movwf	PWMH

	movf	PRO,W
	movwf	PWM

ICom	movf	INTH,W	;add in integral term
	addwf	PWMH,F

	movf	INT,W
	addwf	PWM,F
	btfsc	STATUS,C
	incf	PWMH,F

DCom	movf	DIFH,W	;add in differential term
	addwf	PWMH,F

	movf	DIF,W
	addwf	PWM,F
	btfsc	STATUS,C
	incf	PWMH,F

Ovrld	btfsc	PWMH,7	;did PWM go negative?
	goto	NegPwm	;yes
	bcf	PORTB,6	;no - turn off overload LED
	goto	PwmGen

NegPwm	bsf	PORTB,6	;turn on overload LED
	clrf	PWMH	;set PWM to 0
	clrf	PWM

;****************************************************************************
;*	PwmGen	Divides PWHM:PWM by 8 (3 right shifts)
;*		2 LSbits of PWM sent to 2 LSbits of duty cycle register
;*		remaining 6 bits sent to CCPR1L (duty cycle register)
;*		
;*		A/D has been aquiring VUNREG, start conversion.
;*		
;*	Input Variables:
;*		PWMH:PWM	PWM drive
;****************************************************************************

PwmGen
	rrf	PWMH,F	;PWMH
	rrf	PWM,F	;PWM

	rrf	PWMH,F	;PWMH
	rrf	PWM,F	;PWM

	rrf	PWMH,F	;PWMH - can ignore contents of PWMH now
	rrf	PWM,F	;PWM

	bcf	CCP1CON,4	;clear ls bits of PWM duty cycle
	bcf	CCP1CON,5

	rrf	PWM,F	;shift carry INTo PWM, lsbit INTo carry
	btfsc	STATUS,C	;is carry set?
	bsf	CCP1CON,4	;set PWM duty cycle lsb

	rrf	PWM,F	;shift carry INTo PWM, lsbit INTo carry
	btfsc	STATUS,C	;is carry set?
	bsf	CCP1CON,5	;set PWM duty cycle lsb

	movf	PWM,W	;get PWM
	andlw	0x3f	;mask off 2 ms bits (78.125KHz)
	movwf	CCPR1L	;put in PWM duty cycle

	bsf	ADCON0,2	;start conversion VUNREG

;****************************************************************************
;*	up/dn buttons	
;*		Debounces UP and DOWN buttons
;*			Increments counters once for each pass through
;*			the loop when button pressed.
;*		
;*			If button not pressed, counter is reset.
;*		
;*			If a button is successfully debounced, 
;*			both counters are reset.
;*			
;*			Debounce delay is about 0.5 seconds
;*		
;*		Moves voltage indicator bit as required.
;*		
;*		Finds index into lookup table, and calls table.
;*		
;*		Saves result from lookup table as new SETPOINT
;*		
;*	Input Variables:
;*		PORTB	Current indicator data
;*		DNCH:DNCL	Down button debounce counter
;*		UPCH:UPCL	Up button debounce counter
;*			
;*	Output Variables:
;*		PORTB	Current indicator data
;*		DNCH:DNCL	Down button debounce counter
;*		UPCH:UPCL	Up button debounce counter
;*		SETPOINT	New voltage setpoint
;****************************************************************************

	movf	PORTB,W	;move LED data to temp space
	andlw	0x3f	;mask off non-voltage LEDs
	movwf	TEMPA

Dnb	btfss	PORTA,5	;down
	goto	Upb	;down not pushed
	incfsz	DNCL,f	;down is pushed, inc debounce
	goto	Wc3	;no carry - go to next module
	incfsz	DNCH,f	;inc debounce counter high byte
	goto	Wc3	;no carry - go to next module
			;------------------------------- select next lower LED
	bcf	STATUS,C	;select next lower LED
	rrf	TEMPA,F	;shift LED data down 1 voltage
	btfsc	STATUS,C	;3V LED was set before rotate, so
	bsf	TEMPA,0	;set it again

	goto	Dunb		

Upb	btfss	PORTA,2	;up button
	goto	Nob	;up not pushed - no buttons pushed
	incfsz	UPCL,f	;down is pushed, inc debounce
	goto	Wc3	;no carry - go to next module
	incfsz	UPCH,F	;inc debounce counter high byte
	goto	Wc3	;no carry - go to next module
			;-------------------------------- select next higher LED
	bcf	STATUS,C	;select next higher voltage LED
	rlf	TEMPA,F	;shift LED data up 1 voltage
	btfsc	TEMPA,6	;if 9V LED was set before,
	bsf	TEMPA,5	;set it again, and
	bcf	TEMPA,6	;clear the overload LED

Dunb	movf	TEMPA,W	;move LED data back to PORTB
	movwf	PORTB

	clrf	TEMPB
	decf	TEMPB,F	;set TEMPB to -1

NewSet	incf	TEMPB,F	;count up
	rrf	TEMPA,F	;rotate least sig bit INTo carry
	btfss	STATUS,C	;is carry set now?
	goto	NewSet	;no, try again

	movf	TEMPB,W	;yes, put count in w
	call	Tbl	;get corresponding value for PWM
	movwf	SETPOINT	;put value in SETPOINT

Nob	clrf	DNCL	;clear down button debounce counter
	clrf	UPCL	;clear up button debounce counter
	movlw	DEL1
	movwf	DNCH	;preset down button debounce counter high byte
	movwf	UPCH	;preset up button debounce counter high byte

;****************************************************************************
;*	Wc3	VUNREG has been converted by now,
;*		Get result, save in VUNREG.
;*		
;*		Select VOUT to aquire.
;*		
;*		Test VUNREG to see if it has dropped too low.
;*			If too low, call protective shut-down. (dies here)
;*		
;*	Input Variables:
;*		VUNREG	Input voltage.
;****************************************************************************

Wc3
	btfsc	ADCON0,GO_DONE	;test if done
	goto	Wc3	;no, wait some more
	movf	ADRES,W	;get conversion result
	movwf	VUNREG	;save result

	movlw	AVOUT	;select feedback channel to aquire
	movwf	ADCON0

	movf	VUNREG,W	;get UNREG
	sublw	0x50	;10V-VUNREG=?  C=1 if VUNREG<10V
	btfsc	STATUS,C
	goto	ShutDn	;turn off regulator (dies here)


;****************************************************************************
;*	Shift	shift errors - save current error as old
;*		E1H:E1 = E0H:E0
;*		
;*		Go back to top of loop.
;*		
;*	Input Variables:
;*		E0H:E0	Present error
;*		
;*	Output Variables:
;*		E1H:E1	Previous error
;****************************************************************************

Shift
	movf	E0,W
	movwf	E1

	movf	E0H,W
	movwf	E1H

	goto	Again

;****************************************************************************
;*	Tbl	Look up table.
;*		
;*		Called with an index in W.
;*		
;*		Index is added to program counter.
;*		
;*		Execution jumps to "retlw" which will return
;*		to the calling program with table value in w.
;*		
;*	Input Variables:
;*		w	Offset index into table
;*		
;*	Output Variables:
;*		w	Value from table
;****************************************************************************

Tbl				;call with index in w
	addwf	PCL,F		;add index to PC

	dt	0x4d, 0x74, 0x82, 0x9b, 0xc2, 0xea
;output voltage	 3.0   4.5   5.0   6.0   7.5   9.0

;****************************************************************************
;*	ShutDn	Protective shutdown. Entry into this routine is a result
;*		of a low input voltage. This turns off the switching converter
;*		before the series pass transistor can overheat.
;*		
;*		PWM on-time is set to 0, turning off the converter.
;*		
;*		The converter trip LED is lit, indicating shutdown.
;*		
;*		Execution then loops without exit. The only exit is to
;*		reset the controller.
;****************************************************************************

ShutDn
	bcf	CCP1CON,4	;turn off PWM
	bcf	CCP1CON,5
	clrf	CCPR1L

	bsf	PORTB,7	;light trip LED

Dead
	goto	Dead	;stays here


	END

